<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
* @filesource
*/ /** */

require_once 'restricted_controller.php';
require_once APPPATH.'libraries/attachment.php';
require_once APPPATH.'models/mailbox.php';
require_once APPPATH.'models/user.php';


/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
class Inbox extends Restricted_controller {
	var $mailbox; //the current mailbox - either the user's personal mailbox or a group mailbox
	
    function __construct() {
        parent::__construct();
		
		$this->load->model('response_model');
		
		//get group mailbox information
		$group_mailboxes = $this->session->userdata('group_mailboxes');
		
		//currently selected mailbox - personal or group
		$mailbox_group = $this->session->userdata('mailbox_group');
		
		//default to the user's personal mailbox if we don't have a mailbox selected or if no group mailboxes are available or if the mailbox group isn't valid
		if(empty($mailbox_group) || empty($group_mailboxes) || !array_key_exists($mailbox_group, $group_mailboxes) || !Mailbox::exists( array('user_name' => $mailbox_group) )){
			$mailbox_group = $this->session->userdata('username');
		}

		$this->mailbox = Mailbox::find_one( array('user_name' => $mailbox_group) );
		
#TODO - special case for if we still don't have a mailbox? or if mailbox isn't active?

		//set the session mailbox group in case we've changed it
		if($this->session->userdata('mailbox_group') != $mailbox_group)
			$this->session->set_userdata('mailbox_group', $mailbox_group);	

    }

    /* This function changes the current mailbox by setting the current mailbox in session
     * to the provided input. It is called from links, so it expects a url encoded base64 string,
     * which it will then decode. It redirects back to the index when done.
     */
    public function change_mailbox($mailbox) {
		$this->session->set_mailbox_location(base64_decode(urldecode($mailbox))); //set current mailbox
		redirect('inbox'); //redirect to index
    }
	
	/* This function allows the user to change their currently selected mailbox group */
	public function change_mailbox_group($group) {
		$group = rawurldecode($group);
		$this->session->set_userdata('mailbox_group',$group);
		if($this->session->userdata('mailbox_group') != $this->session->userdata('username')) {
			$ldapconfig['user'] = $this->session->userdata('username');
			$ldapconfig['pwd'] = $this->encrypt->decode($this->session->userdata('ep'));
			$this->load->library('ldap',$ldapconfig);
			$ldap_lookup = $this->ldap->search(NULL,1,array('cn'),'(&(ObjectClass=groupOfNames)(ou=' . $group . '))',LDAP_BASE_DOMAIN);
			$this->session->set_userdata('mailbox_group_cn',$ldap_lookup[0]['cn']);			
		}
		else {
			$this->session->set_userdata('mailbox_group_cn','Personal (' . $this->session->userdata('username') . ')');
		}
		$this->session->set_mailbox_location('inbox');
		redirect('inbox');
	}	


	public function close_authorize() {
		$this->output->append_output('<script type="text/javascript">parent.$.fancybox.close();</script>');
	}
  

   /* This function checks to see which type of form was submitted so that forms with multiple buttons can be handled  
    * differently
    */
    public function form_check() {
        $post_array = $this->input->post(NULL,TRUE);

        if(!empty($post_array)) {
            $send = array_key_exists('send',$this->input->post(NULL,TRUE)); //detect send submission
            $save = array_key_exists('save',$this->input->post(NULL,NULL,TRUE)); //detect save submission
            $archive = array_key_exists('archive',$this->input->post(NULL,TRUE)) || array_key_exists('archive_hidden',$this->input->post(NULL,TRUE)) ; //detect archive submission
            $compose = array_key_exists('compose',$this->input->post(NULL,TRUE)); //detect compose submission
			$create_folder = array_key_exists('create_folder',$this->input->post(NULL,TRUE)); //detect create folder submission
			$mark_as_read = array_key_exists('mark_as_read',$this->input->post(NULL,TRUE)) || array_key_exists('mark_as_read_hidden',$this->input->post(NULL,TRUE)) ; //detect mark as read submission
            $mark_as_unread = array_key_exists('mark_as_unread',$this->input->post(NULL,TRUE)) || array_key_exists('mark_as_unread_hidden',$this->input->post(NULL,TRUE)) ; ; //detect mark as unread submission
			$feedback = array_key_exists('feedback',$this->input->post(NULL,TRUE)); //detect feedback submission
			//detect move submission (either from hidden move button or regular one)
            if(array_key_exists('move',$this->input->post(NULL,TRUE))) {
				$move = TRUE; $move_action = 'move';
			}
			else if(array_key_exists('move_hidden',$this->input->post(NULL,TRUE))) {
				$move = TRUE; $move_action = 'move_hidden'; $this->session->set_flashdata('last_button','move_hidden');
			} else { $move = FALSE; }
			
            //detect rename folder submission (submission only happens via button on hidden menu)
			foreach($post_array as $key => $input) {
				if(strpos($key,'rename_folder_') !== FALSE) {
					$folder_name = rawurlencode(base64_encode($this->input->post('old_value_'.str_replace('rename_folder_','',$key),TRUE)));
				}
			}
			
			//based on detections, call appropriate functions
            if($send || $save){$this->compose();/*moving this functionality to the compose method so we can redisplay values from post if there are errors; ultimately, would like to bypass form_check and just use compose*/}
            else if($compose) { redirect('inbox/compose'); }
			else if($create_folder) { $this->create_folder(); }
            else if($move) { $this->move($move_action); }
            else if($mark_as_read) { 
				if(array_key_exists('mark_as_read_hidden',$this->input->post(NULL,TRUE))) { $this->session->set_flashdata('last_button','mark_as_read_hidden'); }
				$this->mark_as_read(); 
			}
			else if($mark_as_unread) { 
				if(array_key_exists('mark_as_unread_hidden',$this->input->post(NULL,TRUE))) { $this->session->set_flashdata('last_button','mark_as_unread_hidden'); }
				$this->mark_as_unread(); 
			}
            else if($feedback) { $this->feedback(); }
			else if(isset($folder_name)) { $this->rename_folder($folder_name); redirect('inbox'); }
			
			//for archive perform some additional actions
            else if($archive) {
/* NOTE - This code doesn't appear to allow for individually archived message (from the message view page), and the success/fail message will only reflect the most recently archived message, not the full array of messages that were archived.  We may see both an error message and a success message if only one message fails.  We should consider refactoring this at some point.  -- MG 2014-05-06 */
				if(array_key_exists('archive_hidden',$this->input->post(NULL,TRUE))) { $this->session->set_flashdata('last_button','archive_hidden'); }
				foreach($this->input->post(NULL, TRUE) as $key => $value) { //loop through checkboxes to find messages to archive
					if($value == 'on') { 
						$id = str_replace('select','',$key);
						$this->archive($id, $accessed_from_form_check = TRUE);
					}
				}
                redirect('inbox');
			}
            else { show_404(); }
        }
        else { 
            //catch upload size > post_max_size error
            if(empty($_FILES) && empty($_POST) && isset($_SERVER['REQUEST_METHOD']) && strtolower($_SERVER['REQUEST_METHOD']) == 'post'){
                $this->session->set_userdata('message',
                '<script>
					var nth = noty({
						text: \'Failed to send message, attachment size was greater than 10 MB. Failed to save draft since message size was greater than server request limit.\',
						type: \'error\',
						timeout: 5000
					});
                </script>'); 
                redirect('inbox'); 
            }
            else {
                $this->session->set_userdata('message',
                '<script>
					var nth = noty({
						text: \'Form submission failed.\',
						type: \'error\',
						timeout: 5000
					});
                </script>');
                redirect('inbox');
            }
        }
    }
	
	/* This searches the address book and contacts list and returns JSON encoded results for the auto-complete */
	public function get_contacts_search($input = null, $function = false, $contain_all = null){  
		$input = urldecode($input);
		if(is_null($input)) { $input = ""; }
		$result_arr = array();
		$added_addresses = array();
		//get addresses from global address book in LDAP
		$ldapconfig['user'] = $this->session->userdata('username');
		$ldapconfig['pwd'] = $this->encrypt->decode($this->session->userdata('ep'));
		$this->load->library('ldap',$ldapconfig);
		if(is_null($contain_all)) { $contain_all = true; }
		$entries = ($this->ldap->search($input,NULL,NULL,NULL,NULL,$contain_all));
		foreach($entries as $key => $val) {
			if($this->valid_formatted_address($val['mail'])) {
				$contact_arr = array('name'=>(isset($val['displayname'])?$val['displayname']:"Undefined"),'id'=>$val['mail']);
				array_push($result_arr,$contact_arr);
				array_push($added_addresses,$contact_arr['id']);
			}
		}
		if(!$function){
			//get global distribution lists
			$list_result = $this->public_distribution_list_model->find(array('cn'=>$input.'*'));
			
			foreach($list_result as $index => $entry) {
				$address_tokens = array();
				$list_result[$index]['id'] = $this->public_distribution_list_model->alias($entry['id']);
				$list_result[$index]['description'] = '(Global) '.$list_result[$index]['description'];
				
				if(!empty($list_result[$index]['addresses'])){
					//get display names and addresses for distribution list expansion
					$address_array = preg_split("/;/", $list_result[$index]['addresses'], null, PREG_SPLIT_NO_EMPTY);
					$display_names_array = $this->distribution_list_model->display_names_for_direct_addresses($address_array);
					foreach($address_array as $address) {
						if(!isset($display_names_array[$address])) {
							$display_names_array[$address] = trim($address);
						}
					}
					foreach($display_names_array as $address => $display_name) {
						array_push($address_tokens,$display_name . ' (' . trim($address) . ')');
					}
					
					$display_names = implode('; ', $display_names_array);
					$addresses_in_display_order = implode('; ', array_keys($display_names_array));
					$list_result[$index]['display_names'] = $display_names;
					$list_result[$index]['addresses_in_display_order'] = $addresses_in_display_order;
					$list_result[$index]['address_tokens'] = json_encode($address_tokens);
					array_push($result_arr,$list_result[$index]);
				}
			}
		}
		
		//get contacts from contact list in database
        $contacts = $this->db->query('SELECT * FROM contacts WHERE (last_name LIKE (' . $this->db->escape($input) . "+ '%') OR first_name LIKE (" . $this->db->escape($input) . "+ '%') OR middle_name LIKE (" . $this->db->escape($input) . "+ '%') OR mail LIKE (" . $this->db->escape($input) . "+ '%')) AND user_id = (SELECT user_id FROM users WHERE  user_deleted_flag=0 AND user_name = " . $this->db->escape($this->session->userdata("username")) . ")"); 
		for($i = 0; $i < $contacts->num_rows(); $i++) {
			$row = $contacts->row_array($i);
			$displayname = $row['last_name'] . ', ' . $row['first_name'];
			if(isset($row['middle_name']) && (strlen($row['middle_name']) > 0)) { $displayname .= ' ' . $row['middle_name']; }
			
			if($this->valid_formatted_address($row['mail'])){
				$contact_arr = array('name' => $displayname, 'id' => $row['mail']);
				//don't return blank/null attributes
				foreach($contact_arr as $attr => $val) {
					if(!isset($val) || strlen($val) <= 0) {
						unset($contact_arr[$attr]);
					}
				}
				if($function){
					$wasadded = array_search($contact_arr['id'],$added_addresses);
					if (!($wasadded === 0 || $wasadded > 0 )){
						array_push($result_arr,$contact_arr);
						array_push($added_addresses,$contact_arr['id']);
					}
				}
				else{
					array_push($result_arr,$contact_arr);
				}
			}
		}
		//get any contacts from the admin panel contact list that are shared with this user
		if ($this->user->is_admin) {
			$admin_contacts = $this->db->query("SELECT * FROM admin_contact_list WHERE (last_name LIKE (" . $this->db->escape($input) . "+ '%') OR middle_name LIKE (" . $this->db->escape($input) . "+ '%') OR first_name LIKE (" . $this->db->escape($input) . "+ '%') OR direct_address LIKE (" . $this->db->escape($input) . "+ '%'))");
		}
		else{
			$admin_contacts = $this->db->query("SELECT * FROM admin_contact_list WHERE (last_name LIKE (" . $this->db->escape($input) . "+ '%') OR middle_name LIKE (" . $this->db->escape($input) . "+ '%') OR first_name LIKE (" . $this->db->escape($input) . "+ '%') OR direct_address LIKE (" . $this->db->escape($input) . "+ '%')) AND sharing LIKE ('%" .$this->session->userdata("username"). "%')");
		}
		for($i = 0; $i < $admin_contacts->num_rows(); $i++) {
			$row = $admin_contacts->row_array($i);
			$displayname = $row['last_name'] . ', ' . $row['first_name'];
			if(isset($row['middle_name']) && (strlen($row['middle_name']) > 0)) { $displayname .= ' ' . $row['middle_name']; }
			$admin_contact_arr = array('name' => $displayname, 'id' => $row['direct_address']);
			//don't return blank/null attributes
			foreach($admin_contact_arr as $attr => $val) {
				if(!isset($val) || strlen($val) <= 0) {
					unset($admin_contact_arr[$attr]);
				}
			}
			array_push($result_arr,$admin_contact_arr);
		}
		
		if(!$function){
			//get private distribution lists and load into results array
			$this->db->order_by('name');
			$this->db->like('name',$input,'after');
			$dist_list_result = $this->private_distribution_list_model->find(array());
			for($i = 0; $i < count($dist_list_result); $i++) {
				$address_tokens = array();
				$row = $dist_list_result[$i];
				$list_arr = array('name'=>$row['name'],'id'=>$this->private_distribution_list_model->alias($row['id']),'description'=>'(Private) '.$row['description']);
				//don't return blank/null attributes
				foreach($list_arr as $attr => $val) {
					if(!isset($val) || strlen($val) <= 0) {
						unset($list_arr[$attr]);
					}
				}
				if(!empty($row['addresses'])){
					//get display names and addresses for distribution list expansion
					$address_array = preg_split("/;/", $row['addresses'], null, PREG_SPLIT_NO_EMPTY);
					$display_names_array = $this->distribution_list_model->display_names_for_direct_addresses($address_array);
					foreach($address_array as $address) {
						if(!isset($display_names_array[$address])) {
							$display_names_array[$address] = trim($address);
						}
					}
					foreach($display_names_array as $address => $display_name) {
						array_push($address_tokens,$display_name . ' (' . trim($address) . ')');
					}
					$display_names = implode('; ', $display_names_array);
					$addresses_in_display_order = implode('; ', array_keys($display_names_array));
					$list_arr['display_names'] = $display_names;
					$list_arr['addresses_in_display_order'] = $addresses_in_display_order;
					$list_arr['address_tokens'] = json_encode($address_tokens);
					array_push($result_arr,$list_arr);
				}
			}
		}
		//sort combined result array (natural order, case insensitive)
		usort($result_arr, function( $el1, $el2) { return strnatcasecmp( $el1['name'], $el2['name']); });
		
		//add whatever the user is currently typing so that it is allowed as well
		if($this->valid_formatted_address($input)) {
			$contact_arr = array('name' => $input, 'id' => $input);
			array_push($result_arr,$contact_arr);
		}
		array_walk_recursive($result_arr, function (&$value) {
			$value = htmlentities($value);
		});
		//only echo if the request comes from AJAX
		if($contain_all){
			$result_arr = array_slice($result_arr, 0,10);
		}

		if(IS_AJAX && !$function) {
			echo json_encode($result_arr);
		}
		return $result_arr;
	}	

    /* This is the main function for the inbox. It will get the header data for all messages in the current mailbox
     * which can then be displayed in a table in the inbox/index view.
     */
    public function index() {    		
		$this->load->model('flag_model');
		if($this->mailbox->is_group) { $this->load->model('workflow_model'); }
	
		$current_user = $this->user->id;
		$display_name = $this->user->cn;
		$folder = strtolower(element('folder', $_SESSION, 'inbox'));
		$folder_name = strip_from_beginning(CUSTOM_MAILBOX_PREFIX, element('folder_name', $_SESSION, 'Inbox'));
		$show_workflow = ($this->mailbox->is_group && isset($this->workflow_model) && !in_array($folder, array('archived', 'draft', 'sent')));
		$title = PORTAL_TITLE_PREFIX.$folder_name;

		//SET UP TEMPLATE VARIABLES	
		$mailbox_list_data = $this->mailformat->mailbox_list();
		$this->template->set('mailboxes', $mailbox_list_data['folder_list']);
		$this->template->set('deepest_tree_level', $mailbox_list_data['deepest_tree_level']);
		$this->template->set('folder_search_select_menu', $mailbox_list_data['folder_search_select_menu']);
		$this->template->set('json_mailbox_list', $this->json_folder_list(FALSE));
		$this->template->set('page_title', 'Message List');
		$this->template->set('form_destination', 'inbox/form_check');
		$this->template->set('menu_partial', 'inbox/_inbox_menu');
		
		if($this->mailbox->is_group){
			$this->template->set('other_members', $this->get_group_members($this->mailbox->name));
		}
		
		//check to see what message we should start with
		$page_start = element('page_start', $_SESSION, 1);
		if(!$this->is->nonzero_unsigned_integer($page_start)){
			$page_start = 1;
			$_SESSION['page_start'] = $page_start;
		}
							
		//FIND THE MESSAGES	
		$limit = $display_per_page = INBOX_DISPLAY_PER_PAGE; //configure how many emails to display per page 
		$start = $page_start - 1;
		$order_by = $sort_column = element('sort_column', $_SESSION);
		$order = $sort_direction = element('sort_direction', $_SESSION);
		
		$criteria = array_filter(compact('folder', 'limit', 'order', 'order_by', 'start'), 'strlen');
		$criteria['part'] = 'headers'; 
		
		if($order_by == 'flag' || $order_by == 'workflow'){
			unset($criteria['order_by'], $criteria['order']);  //the API can't sort these - we'll sort them lwhen we get the messages back
		}
		$filter_folder = element('filter_folder', $_SESSION);
		if($folder === 'archived' && in_array($filter_folder, array('inbox','sent','draft'))){
			$filter = array('original_folder' => $filter_folder);
			$criteria['filter'] = rawurlencode(base64_encode(json_encode($filter)));
		}		
		$messages = $this->mailbox->messages( $criteria  );

		//if we have any problems retrieving the messages, show the error right away before calculating all the other things		
		if($this->api->http_status == 403 && $this->api->message() == 'Access denied. The application is not authorized for this action.'){
			return $this->template->load('inbox/template', 'inbox/_service_permission_error');
		}elseif($this->api->http_status != 200 || !empty($this->api->format_errors)){
			return $this->template->load_string('inbox/template', 'Unable to retrieve messages at this time. If the problem persists, please contact a system administrator.');
		}
		
		//if there are no messages in this location, tell us that.
		if(empty($messages)){
			if($filter_folder === 'draft') { $filter_folder = 'drafts'; } //pluralize drafts
			$data = compact('current_user', 'display_name', 'folder_name', 'show_workflow', 'title', 'filter_folder');
			$this->template->set($data);
			return $this->template->load_string('inbox/template', 'There are no messages to display.');
		}
		
		$total = element('total', $this->api->output_as_array());
		$total_unseen = $_SESSION['unseen'] = element('total_unseen', $this->api->output_as_array());
		$message_count = count($messages);
		if(!empty($_SESSION['unseen'])) {
			$title .= ' ('.$_SESSION['unseen'].')';
		}
		if($filter_folder === 'draft') { $filter_folder = 'drafts'; } //pluralize drafts
		$data = compact('current_user', 'display_name', 'folder_name', 'show_workflow', 'title', 'filter_folder');
		$this->template->set($data);
		//if we have an invalid page_start value, set it to something sensible if po ssible and start this script over again
		if($this->is->nonzero_unsigned_integer($total) && $page_start > $total){
			if($total > INBOX_DISPLAY_PER_PAGE)
				$_SESSION['page_start'] =  $total - INBOX_DISPLAY_PER_PAGE;
			else
				unset($_SESSION['page_start']);			
			redirect(current_url());
		}
		$this->template->set(compact('page_start', 'message_count', 'total'));
		$custom_flags = $this->flag_model->find_for_messages(array_keys($messages));
		foreach($custom_flags as $custom_flag){
			$messages[$custom_flag['message_id']]->flag = $custom_flag;
		}
		
		if($show_workflow){
			$workflow_items = $this->workflow_model->find_for_messages(array_keys($messages));
			foreach($workflow_items as $workflow_item){
				$messages[$workflow_item['message_id']]->workflow_item = $workflow_item;
			}
		}
		
		//SET UP THE SORT ICONS
		$sort_icon = '';
		if($sort_direction == 'asc') {  
			$sort_icon = ' <span class="sort_icon">&#x25b2;</span>'; 
		}elseif($sort_direction == 'desc') { 
			$sort_icon = ' <span class="sort_icon">&#x25bc;</span>'; 
		}
		$data = $data + compact('sort_icon', 'sort_column', 'sort_direction');	
				

		//handle special sort situations
		if($sort_column == 'flag') {
			usort($messages,'cmp_flag');
		}
		elseif($sort_column == 'workflow') {
			usort($messages,'cmp_workflow');
		}
		
		
		//set headers within data to pass to view
		$data['headers'] = $data['messages'] = $messages;
		$data['show_workflow'] = $show_workflow;
		
		//get last login data
		$get_last_logon = $this->user_model->last_login_info();
		$row = $get_last_logon->row_array(1);
		if($row['success'] == TRUE) { $success = 'successful'; } else { $success = "<span style=\"font-weight: bold;\">unsuccessful</span>"; }
		$data['first_login_message'] = '<div id="logon_notice" class="modal_msg">';
		$data['first_login_message'] .= 'Last login attempt was ' . $success . ' from IP address: ' . $row['ip_address'] . ' at ' . date('m/d/y h:i:s A',$row['login_time']);
		$data['first_login_message'] .= '<a onclick="$(\'.modal_msg\').delay(1000).fadeOut(1000,function() { $(\'.modal_msg\').remove(); });"><img src="/jscripts/fancybox/fancy_close.png" alt="Close Login Message" /></a>';
		$data['first_login_message'] .= '</div>';
		if($get_last_logon->num_rows === 1){ //first time logging in
			$data['first_login_message'] .='<script>$().ready(function(){getWebservicePermission(\"by_mailbox_group\");});</script>';
		}
		
		if($folder == 'sent' || $folder == 'draft'){			
			return $this->template->load('inbox/template', 'inbox/_messages_sent_draft', $data);
		}
		
		$this->template->load('inbox/template', 'inbox/_messages', $data);		
    }
	
     /* Provides number of messages in inbox since the index was last accessed. For use with ajax request to see
      * if new messages are waiting.
      */
    public function message_waiting_count() {
		$this->action_is_not_user_activity(); 
		
		$count = 0;
		$folders = $this->mailformat->get_folder_list()->response->folder;
		$current_folder = element('folder', $_SESSION, 'inbox');
		if(!(isset($_SESSION['filter_folder']) && $current_folder === 'archived')){ // being filtered in archive
		
			foreach($folders as $folder) {
				if($folder->id === $current_folder) {
					$count = $folder->new;
				}
			}
			if(isset($_SESSION['unseen'])) {
				echo $count - $_SESSION['unseen']; //don't show count with the unseen messages already displayed
				$_SESSION['unseen'] = $count;
			}
			else { echo $count; }
		}
		else{
			echo 0;
		}
    }

    /* This function changes the page start session value. The page start session value determines
     * the message id that the index function will start the current page display at.
     * TO DO: Add input validation. Low priority since invalid call will simply not return messages.
     */
    public function page($start) {
        $_SESSION['page_start'] = $start;
        redirect('inbox');
    }
	 /*This functions changes the the filter for the archive
	 */
	public function filter_archive($folder){
		if(in_array($folder, array('inbox','sent','draft'))){ $_SESSION['filter_folder'] = $folder; }
		else { unset($_SESSION['filter_folder']); }
		redirect('inbox');
	}
	
	public function sort($item) {
		//set sort column (if valid)
		$valid_sort_columns = array('workflow', 'flag', 'priority', 'timestamp', 'id', 'sender', 'to', 'cc', 'bcc', 'attachments', 'size', 'subject', 'plain', 'status_code');
		$same_column = FALSE;
		if(in_array($item, $valid_sort_columns)) {
			if(isset($_SESSION['sort_column']) && $_SESSION['sort_column'] === $item)
				$same_column = TRUE;
			$_SESSION['sort_column'] = $item;
		}
		//set sort direction
		if(!isset($_SESSION['sort_direction'])) { 
			$_SESSION['sort_direction'] = 'asc'; 
		}else {
			if($_SESSION['sort_direction'] === 'asc' && $same_column) { 
				$_SESSION['sort_direction'] = 'desc'; 
			}else if($_SESSION['sort_direction'] === 'desc' && $same_column) { 
				unset($_SESSION['sort_direction']); 
				unset($_SESSION['sort_column']);
			}
			else { 
				$_SESSION['sort_direction'] = 'asc'; 
			}
		}
		redirect('inbox');
	}
		
    /* This uses imap_search() to search the imap server for messages relating to the search term.
     * This function is currently implemented to check the subject, text, to and from fields only.
     * imap_search() is limited to IMAP2 search parameters (meaning no OR statement), so we have to perfrom
     * a seperate search for each criteria and then combine results.
     */
#KNOWN ISSUE - Sorting messages clears the search
    function search($advanced_search = FALSE) {
        $this->form_validation->set_rules('search_input','Search Input','xss_clean');
        if($this->form_validation->run() === TRUE) {
			$current_user = $this->user->id;
			$display_name = $this->user->cn;
			$mailbox_str = element('folder_name', $_SESSION, 'Inbox');
			$title = 'Searching '.$mailbox_str;
			$search_input = $this->input->post('search_input',TRUE);
			//advanced search filters
			$from = $this->input->post('from',TRUE);
			$recipient = $this->input->post('recipient',TRUE);
			$subject_title = $this->input->post('subject_title',TRUE);
			$plain_message = $this->input->post('plain_message',TRUE);
			$priority = $this->input->post('priority',TRUE);
			$original_folder = $this->input->post('original_folder',TRUE);
			$time_frame = $this->input->post('time_frame',TRUE);
			$folders_to_search = $this->input->post('folder_search_select',TRUE);
			$is_multuple_folder_search = false;
			if(sizeof($folders_to_search) > 1) {
				$is_multuple_folder_search = true;
			}
			$mailbox_list = $this->mailformat->mailbox_list($is_multuple_folder_search);
			$this->template->set('title', PORTAL_TITLE_PREFIX.$title);
			$this->template->set('mailboxes', $mailbox_list['folder_list']);
			$this->template->set('folder_search_select_menu', $mailbox_list['folder_search_select_menu']);
			$this->template->set('page_title', 'Message List');
			$this->template->set('form_destination', 'inbox/form_check');
			$this->template->set('menu_partial', 'inbox/_inbox_menu');

			//check to see what message we should start with
			$page_start = element('page_start', $_SESSION, 1);
			if(!$this->is->nonzero_unsigned_integer($page_start)){
				$page_start = 1;
				$_SESSION['page_start'] = $page_start;
			}
			
			//FIND THE MESSAGES	
#DON'T APPLY PAGINATION FOR NOW - SWITCHING PAGES CLEARS THE SEARCH
/*			$limit = $display_per_page = INBOX_DISPLAY_PER_PAGE; //configure how many emails to display per page 
			$start = $page_start - 1; */
			$order_by = $sort_column = element('sort_column', $_SESSION); //don't allow sorts for now - sorting gets us out of the search
			$order = $sort_direction = element('sort_direction', $_SESSION);
			$folders_to_search_str = '';
			$folders_to_search_str_for_display = '';
			if($advanced_search) {
				//set up original folder filter (in Archive mailbox only)
				$original_folder_str = '';
				if($original_folder !== false){
					foreach($original_folder as $value) {
						if(strpos($value,'all') === false) {
							$original_folder_str .= $value . ',';
						}
					}
				}
				else {
					$original_folder_str = '';
				}
				$original_folder_str = replace_last_with(',', '', $original_folder_str);
				$search_words = array(
						'contains_sender' => $from,
						'contains_recipients' => $recipient,
						'original_folder' => $original_folder_str,
						'contains_subject' => $subject_title,
						'contains_plain' => $plain_message
				);
				$filters = array();
				foreach ($search_words as $key=>$value){
					if($value && !empty($value)){
						$filters[$key] = $value;
					}	
				}
				if(!empty($subject_title)){
					$filters['contains_subject'] = $subject_title;
				}
				if(!empty($plain_message)){
					$filters['contains_plain'] = $plain_message;
				}
				if(is_numeric($priority)){
					$filters['priority'] = $priority;
				}
				if(is_numeric($time_frame)){  //check is a number.  Can be zero for today
						$filters['first_date'] = strtotime('-'.$time_frame.' days midnight');
				}
				$filter = rawurlencode(base64_encode(json_encode($filters)));	
						
				$system_folder_names = array('inbox'=>'Inbox', 'sent'=>'Sent', 'draft'=>'Drafts', 'archived'=>'Archive');
				if(!empty($folders_to_search)) {
					foreach($folders_to_search as $folder_to_search) {
						$selected_values = explode('_', $folder_to_search);
						if(sizeof($selected_values) == 1) {
							$folders_to_search_str .= $selected_values[0] . ',';
							$folders_to_search_str_for_display .= $system_folder_names[$selected_values[0]] . ', ';
						}
						else {
							$folders_to_search_str .= $selected_values[0] . ',';
							$folders_to_search_str_for_display .= $selected_values[1] . ', ';
						}
					}
					$folders_to_search_str = replace_last_with(',', '', $folders_to_search_str);
					$folders_to_search_str_for_display = replace_last_with(', ', '', $folders_to_search_str_for_display);
				}
				
			}
			else {
				if(empty($search_input))
					$search_input = '*';
				$filter = rawurlencode(base64_encode(json_encode(array( 'fuzzy' => $search_input))));
			}
			
			$folder = element('folder', $_SESSION, 'inbox');
			if($folders_to_search_str !== '' && !empty($folders_to_search_str)) {
				$folder = $folders_to_search_str;
			}
			
			$part = 'headers';
			$headers = $this->mailbox->messages( array_filter(compact('filter', 'folder', 'limit', 'order', 'order_by', 'part', 'start'), 'strlen') );
			$this->template->set('folder_name', element('folder_name', $_SESSION, 'Inbox'));
		
			//if we have any problems retrieving the messages, show the error right away before calculating all the other things		
			if($this->api->http_status == 403 && $this->api->message() == 'Access denied. The application is not authorized for this action.'){
				return $this->template->load('inbox/template', 'inbox/_service_permission_error');
			}elseif($this->api->http_status != 200 || !empty($this->api->format_errors)){
				return $this->template->load_string('inbox/template', 'Unable to retrieve search results. Please contact an administrator if the problem persists.');
			}
			
			if(empty($headers)){
				return $this->template->load_string('inbox/template', 'The search did not yield any results.');
			}
			
			$message_count = count($headers);
			$total = element('total', $this->api->output_as_array(), $message_count);
			
			$sort_icon = '';
			if($sort_direction == 'asc') {  
				$sort_icon = ' <span class="sort_icon">&#x25b2;</span>'; 
			}elseif($sort_direction == 'desc') { 
				$sort_icon = ' <span class="sort_icon">&#x25bc;</span>'; 
			} 
				

			$data = compact('current_user', 'display_name', 'headers', 'message_count', 'page_start', 'sort_column', 'sort_direction', 'sort_icon', 'total','original_folder');
			$data['is_multiple_folder_search'] = $is_multuple_folder_search;
			if($folders_to_search_str_for_display !== '') {
				$mailbox_str = $folders_to_search_str_for_display;
				$data['folders'] = $mailbox_list['folders'];
			}
			if(!$advanced_search) {
				$this->template->set('search_message', 'Found ' .$total. ' '.pluralize_if_necessary('result', $total).' in ' . $mailbox_str . ' for "' . htmlentities($this->input->post('search_input',TRUE)) . '"');
			}
			else{
				$this->template->set('search_message', 'Found ' .$total. ' '.pluralize_if_necessary('result', $total).' in ' . $mailbox_str);
			}
			//load correct template for the folder we are in
			if($folder == 'sent' || $folder == 'draft'){
				return $this->template->load('inbox/template', 'inbox/_messages_sent_draft', $data); 
			}
			$this->template->load('inbox/template', 'inbox/_messages', $data);
		}
		else{
			redirect('/inbox');
		}
    }

	public function valid_formatted_address($str) {
		$valid_format = (!preg_match("/^([a-z0-9\+_\-]+)(\.[a-z0-9\+_\-]+)*@([a-z0-9\-]+\.)+[a-z]{2,6}$/ix", $str)) ? FALSE : TRUE;
		return $valid_format;
	}

	//$str should be a semi-colon separated list of addresses
	public function valid_trusted_address($address) {
		$address_array = explode(';',base64_decode(rawurldecode($address)));
		$address_array = $this->check_dist_list_before_sending($address_array); //check for distribution lists
		$address = rawurlencode(base64_encode(implode(';',$address_array)));

		$this->api->resource = 'direct/validate';
		$this->api->data = compact('address'); 
		$this->api->call();
		echo $this->api->raw_output; //echo so that js can use this
		return (bool)element('valid', $this->api->output_as_array());
	}
	
	
	/*Displays the feedback form view, meant for use in a modal window or pop-up */
	public function feedback_form() {
		$this->load->view('inbox/feedback_form');
	}
	
	/*Displays the feedback form view, meant for use in a modal window or pop-up */
	public function assign_search_menu() {
		$this->load->view('inbox/assign_message');
	}
	
	/* This function grabs the email address from the Get request and passes to the compose function on the flash data,
	 * so, that we do not have to display the link on the url
     */
	public function redirectCompose(){
		$mail = $_GET['mail'];
		$this->session->set_flashdata('mail', $mail);
		//Redirect to Compose
		redirect('/inbox/compose/');
	}
		
	public function search_group_members($search_term = NULL) {
		$group_name = $this->session->userdata('mailbox_group');
		$members = $this->get_group_members($group_name);
		$search_term = strtolower(rawurldecode($search_term));
		if(!is_null($search_term)) { 
			foreach($members as $index => $member) {
				if(!string_begins_with($search_term,strtolower($member['name'])) && !string_begins_with($search_term,strtolower($member['cn']))) {
					unset($members[$index]);
				}
			}
			$members = array_values($members);
		}
		//only echo if the request comes from AJAX
		if(IS_AJAX) {
			echo json_encode($members);
		}
	}

//////////////////////////////////////////////////////////
// THINGS I WOULD LIKE TO MOVE TO THE FOLDER CONTROLLER 
/////////////////////////////////////////////////////////
	/** Creates a folder from the submitted create folder form data. */
    public function create_folder() {
    	$parent_id = $this->input->post('parent_folder_select',TRUE);
        $name = $this->input->post('folder_name',TRUE);
		
		if(empty($name)) {
			$this->session->set_error_message('Failed to create folder. Folder name cannot be blank.');
			return redirect('inbox');
		}
		
		//construct post request
		$fields = array(
			'mailbox' => $this->session->userdata('mailbox_group'),
			'name' => CUSTOM_MAILBOX_PREFIX . $name,
		);
		
		if(Folder::formatted_like_an_id($parent_id))
			$fields['parent_id'] = $parent_id;
		
#TODO - API CALL BELONGS ON THE MODEL, NOT IN HERE		
		if($this->api->call('/direct/folders/create', $fields, 'POST')){
			$this->session->set_success_message('Successfully created the '.$name.' folder.');
		}else{
			$this->session->set_error_message('An error occured and the '.$name.' folder was not created.  Please try again in a moment, and contact an administrator if this problem persists.');
			if($this->api->message() == 'Access denied. The application is not authorized for this action.'){
				$this->session->set_flashdata('service_failure_due_to_permission_message','Failed to create foldr because Manage Direct Service is disabled.');
				$this->session->set_flashdata('service_failure_due_to_permission',true);
				$this->session->set_flashdata('failed_service_name','Manage');
			}			
		}
	
		$_SESSION['mailbox_list'] = $this->mailformat->mailbox_list();
        redirect('inbox');
    }	
    
	/** Prints a folder list in JSON */ //should ideally be in folder controller, but is used by an inbox method
    function json_folder_list($echo = TRUE) {
    	$tree_level = 1;
		$api_folder_list = $this->mailformat->get_folder_list()->response;
		$custom_folder_list = $api_folder_list->folder;
		//if the API failed to return folder list, return false
		if(!isset($api_folder_list) || !isset($api_folder_list->folder)) { return false; }
		
		//if we have folders, pull from the object for looping through
        $folders = $api_folder_list->folder;
		$data = array();
        foreach($folders as $folder) {
			//don't add folders to which messages cannot be moved
			if(($folder->id === 'sent' || $folder->id === 'draft') && $this->session->mailbox_location() === 'archived') {
				$data[] = array('link'=>rawurlencode(base64_encode($folder->id)),'label'=>strip_from_beginning(CUSTOM_MAILBOX_PREFIX, $folder->name));
			}
			else if($folder->id !== 'sent' && $folder->id !== 'draft' && $folder->id !== 'archived') {
				if($folder->id === 'inbox') {
					$data[] = array('link'=>rawurlencode(base64_encode($folder->id)),'label'=>strip_from_beginning(CUSTOM_MAILBOX_PREFIX, $folder->name));
				}
				else {
					if(empty($folder->parent_id)) {
						$this->build_tree_structure($folder->id, $custom_folder_list, $data, $tree_level);
					}
				}
			}
        }
		//only echo if the request comes from AJAX
		if(IS_AJAX) {
			if($echo) { echo json_encode($data); } 
		}
		if(!$echo) { return json_encode($data); }
    }
    
    function build_tree_structure($folder_id, $custom_folder_list, &$data, &$tree_level) {
    	$folder = $custom_folder_list->$folder_id;
    	$indentation = $this->mailformat->get_tree_level_indentation($tree_level);
    	$data[] = array('link'=>rawurlencode(base64_encode($folder->id)),'label'=>$indentation . strip_from_beginning(CUSTOM_MAILBOX_PREFIX, $folder->name));
    	if(isset($folder->child_folders) && sizeof($folder->child_folders) > 0) {
    		$child_folders = $folder->child_folders;
    		$tree_level++;
    		foreach($child_folders as $child) {
    			$this->build_tree_structure($child->id, $custom_folder_list, $data, $tree_level);
    		}
    		$tree_level--;
    	}
    	 
    }
    
//////////////////////////////////////////////////////////
// THINGS I WOULD LIKE TO MOVE TO THE MESSAGE CONTROLLER 
/////////////////////////////////////////////////////////		

#TODO - IDEALLY, THIS SHOULD BE ON THE MESSAGE CONTROLLER, BUT FOR SOME REASON WE CALL IT DIRECTLY IN FORM_CHECK
    /* This function moves mail to the archive mailbox (created on login if non-existent). If the mail is moved          
     * successfully, it is removed from its original mailbox upon expunge. 
     */
    function archive($id, $accessed_from_form_check = false) {
		if(!$this->api->call('direct/messages/archive', array('id' => $id, 'mailbox' => $this->mailbox->name), 'POST')){
			$this->session->set_error_message('Failed to archive message.');
			if($accessed_from_form_check) return false; 
			redirect('inbox/viewmsg/'.$id);
		}
				
		$this->session->set_success_message('Message(s) successfully archived.');
		if($accessed_from_form_check) return true; 
		redirect('inbox');
    }

	
#TODO - IDEALLY, THIS SHOULD BE ON THE MESSAGE CONTROLLER, BUT FOR SOME REASON WE CALL IT DIRECTLY IN FORM_CHECK	
    /* This function moves mail from one mailbox to another
     */
	protected function move($action) {
		if($action === 'move') { $new_folder = rawurldecode(base64_decode($this->input->post('move',TRUE))); }
		else if ($action === 'move_hidden') { $new_folder = rawurldecode(base64_decode($this->input->post('move_select',TRUE))); }
		else { redirect('inbox'); }
		$reason = array();
        foreach($this->input->post(NULL, TRUE) as $key => $value) { 
            if($value == 'on' && $key !== 'select_all') { 
                $uid = str_replace('select','',$key);
				$resource = '/direct/messages/move/format/json';
		
				//construct post request
				$fields = array(
					'mailbox' => $this->session->userdata('mailbox_group'),
					'id' => $uid,
					'folder' => $new_folder,
				);
				$post = '';
				foreach($fields as $key => $value) { $post .= $key.'='.urlencode($value).'&'; } //url-ify the data for the POST
				$post = rtrim($post, '&');
				$response = $this->api_model->webservice_call($resource,'POST',$post)->response;
				$success = isset($response->folder);
				
                if(!$success) {
					if(!empty($response->message)) {
						
						if(string_contains('not originally a draft message', $response->message)) { $reason['not draft'] = 'One or more messages were not originally draft messages and could not be moved to Drafts.'; }
						else if(string_contains('originally a draft message', $response->message)) { $reason['originally draft'] = 'One or more messages were originally draft messages and could not be moved.'; }
						if(string_contains('not originally a sent message', $response->message)) { $reason['not sent'] = 'One or more messages were not originally sent messages and could not be moved to Sent.'; }
						else if(string_contains('originally a sent message', $response->message)) { $reason['originally sent'] = 'One or more messages were originally sent messages and could not be moved.'; }
					}
					else {
						$reason['error'] = 'Could not move message.';
					}
				}
            }
        }
        if(empty($reason)){
        	$this->session->set_success_message('Message(s) successfully moved.');
        }else{
        	$this->session->set_error_message('Failed to move message(s). '.implode("<br/>",$reason));
        }
        redirect('inbox');
    }
		
////////////////////////////////////
// PROTECTED HELPER METHODS
////////////////////////////////////

	protected function check_dist_list_before_sending($addresses) {		
		$addresses = $this->public_distribution_list_model->substitute_addresses_for_aliases($addresses);
		return $this->private_distribution_list_model->substitute_addresses_for_aliases($addresses);
	}
	
	//should be on attachment controller, but called by an inbox method as well as being a public helper method
    protected function clear_attachments() {
        return $this->user->clear_attachment_cache();
    }	
	
    /* Gets form data from feedback modal panel when it is submitted and saves it to the database
     */
    protected function feedback() {
        $this->load->database();
        
        $type = $this->input->post("feedback_type",TRUE);
        if($type == "feedback") { $feedback_type = "General Feedback"; }
        else if($type == "issue") { $feedback_type = "Issue Report"; }
        $comments = $this->input->post("feedback_comments",TRUE);
        $this->load->library("form_validation");
        $this->form_validation->set_rules("feedback_comments","Feedback","max_length[4000]");
		if($this->form_validation->run() === FALSE){			
			show_error("Feedback comments is longer then 4000 characters",400);
		}
        $get_actor_id = $this->db->query("SELECT user_id FROM users WHERE user_deleted_flag = 0 AND user_name=" . $this->db->escape($this->session->userdata("username")));
        if($get_actor_id && $get_actor_id->num_rows() == 1) {
            $row = $get_actor_id->row_array();
            $actor_id = $row["user_id"];
            $feedback = $this->db->query("INSERT INTO feedback (user_id,feedback_type,feedback_comments,feedback_datetime) VALUES (" 
										. $this->db->escape($actor_id) . "," . $this->db->escape($feedback_type) . "," 
										. $this->db->escape($comments) . "," . $this->db->escape(date('U')) . ")");
            if(!$feedback) { show_error('Failed to submit feedback due to internal error.',500); }
        }
        else { show_error('Failed to submit feedback due to internal error.',500); }
    }
	
#TODO - We should be getting this via the API in future
	protected function get_group_members($group_name){
		$dn = LDAP_GROUPS_DN;
		$members  = array();
		//get group information from ldap
		if($this->ldap->connected()) {
			$ldap_result = $this->ldap->search(NULL,NULL,array('cn','ou','description','member'),'(&(ObjectClass=groupofNames)(ou=' . $group_name . '))',$dn);
			if(count($ldap_result) > 0) {
				$data['cn'] = $ldap_result[0]['cn'];
				$data['description'] = isset($ldap_result[0]['description']) ? $ldap_result[0]['description'] : '' ;
				//get group member info from ldap
				if(is_array($ldap_result[0]['member'])) {
					foreach($ldap_result[0]['member'] as $member) {
						$username = str_replace('uid=','',$member);
						$username = str_replace(','.LDAP_ACCOUNTS_DN,'',$username);
						//if($this->user_model->logged_in_user('user_name') !== $username){//remove own username
							//check that user value corresponds to valid user
							$get_user = $this->db->query("SELECT user_id FROM users WHERE user_deleted_flag=0 AND user_name=" . $this->db->escape($username));
							if($get_user) {
								if($get_user->num_rows() == 1) {
									$user_result = $this->ldap->search(null,1,array('cn','displayname'),'(uid='.$username.')');
									$get_user = $get_user->result();
									if(count($user_result) > 0) {
										$member = array('cn' => $user_result[0]['cn'], 'name' => $user_result[0]['displayname'], 'username' => $username, 'id'=>$get_user[0]->user_id);
									}
									else { $member = array('name' => $username, 'username' => $username,'id'=>$get_user[0]->user_id); }
									array_push($members ,$member);
								}
							}
						//}
					}
				}
			}
		}	
		return $members;
	}

    /* This function marks mail as read */
    protected function mark_as_read() {
        foreach($this->input->post(NULL, TRUE) as $key => $value) { 
            if($value == 'on' && $key !== 'selectall') { 
                $uid = str_replace('select','',$key);
				Message::find( array('id' => $uid, 'part' => 'flags', 'mark' => 'read', 'limit' => 1) );		
            }
        }
        redirect('inbox');
    }
	
    /* This function marks mail as unread */
    protected function mark_as_unread() {
        foreach($this->input->post(NULL, TRUE) as $key => $value) { 
            if($value == 'on' && $key !== 'selectall') { 
                $uid = str_replace('select','',$key);
                Message::find( array('id' => $uid, 'part' => 'flags', 'mark' => 'unread', 'limit' => 1) );	
            }
        }
        redirect('inbox');
    }	
	
	protected function _message_from_post(){		
		//FIND OR CREATE THE MESSAGE
		$id = $this->input->post('msg_id',TRUE);
		if(Message::formatted_like_an_id($id)){
			$message = Message::find_one(array('id' => $id, 'sender' => $this->mailbox->email_address()));
			if(!Message::is_an_entity($message)) return $this->error->should_be_a_message_id($id);
		}else{
			$message = new Message( array('sender' => $this->mailbox->email_address()) );
			if(!Message::is_an_entity($message)) return $this->error->warning("I couldn't create a new message for ".$this->mailbox->describe());
		}
		
		//FIND THE VALUES FROM THE FORM
		$values = array( 'original_sender' => $this->user->username,
						 'importance' => $this->input->post('priority', TRUE),
						 'to' => $this->input->post('message_to', TRUE),
						 'cc' => $this->input->post('message_cc', TRUE),
						 'subject' => $this->input->post('message_subject',TRUE),
						 'body' => $this->input->post('message_body', FALSE),
						 'protected_data' =>  $this->input->post('7332_data', FALSE));
			 
		//SET THE FORM VALUES *IFF* THEY'RE DIFFERENT FROM WHAT WE CURRENTLY HAVE (we don't want to make an API call unless we need to)				 				 
		foreach($values as $field => $value){
			if($message->$field != $value)
				$message->$field = $value; 
		}
		
		return $message;
	}	
	
	/**
	* Standard error responses if we're not able to look up a message by id
	* @param Message The result of a Message::find_one() lookup.
	*/
	protected function show_error_if_not_a_message($message){
		if(!Message::is_an_entity($message)){
			if(Message::api()->http_status == 200)
				show_404(); //api call worked, but the message doesn't exist
			elseif(Message::api()->http_status == 403)
				show_error('Access denied. The application is not authorized for this action.');
			else
				show_error('Unable to retrieve message. Please contact an administrator if the problem persists.');
		}
	}
	

}	
/* End of file inbox.php */
/* Location: ./application/controllers/inbox.php */